/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include "OSGConfig.h"
#include "OSGLog.h"

#include <OSGGL.h>
#include <OSGName.h>
#include <OSGStringAttributeMap.h>

#include "OSGNode.h"
#include "OSGFieldContainerPtr.h"
#include "OSGViewport.h"
#include "OSGWindow.h"
#include "OSGCamera.h"
#include "OSGCameraDecorator.h"
#include "OSGProjectionCameraDecorator.h"

OSG_USING_NAMESPACE


/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::Camera
    \ingroup GrpSystemWindowCameras

The Camera base class, see \ref PageSystemWindowCamera for a description.

The only common fields are _sfNear and _sfFar.
*/

/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

void Camera::initMethod (void)
{
}

/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

Camera::Camera(void) :
    Inherited()
{
}

Camera::Camera(const Camera &source) :
    Inherited(source)
{
}

Camera::~Camera(void)
{
}

void Camera::changed(BitVector whichField, UInt32 origin)
{
    Inherited::changed(whichField, origin);
}


/*-------------------------- your_category---------------------------------*/

/*! Setup OpenGL for rendering, call all the necessary commands to start
    rendering with this camera.
*/
void Camera::setup(      DrawActionBase *OSG_CHECK_ARG(action),
                   const Viewport       &port                 )
{

}

/*! Setup OpenGL projection for rendering.
*/
void Camera::setupProjection(      DrawActionBase *OSG_CHECK_ARG(action),
                             const Viewport       &port                 )
{

}

/*! Draw the camera's geometry (if any). Usually there is none.
*/
void Camera::draw(      DrawAction *OSG_CHECK_ARG(action),
                  const Viewport   &OSG_CHECK_ARG(port  ))
{
}

/*! Get/calculate the projection matrix for this camera.
*/
void Camera::getProjection(Matrix &OSG_CHECK_ARG(result),
                           UInt32  OSG_CHECK_ARG(width ),
                           UInt32  OSG_CHECK_ARG(height), int eye, const bool reverseZ)
{
    SFATAL << "Camera::getProjection: NIY" << std::endl;
    abort();
}

/*! Get/calculate the projection translation matrix for this camera. The
    default is identity.
*/
void Camera::getProjectionTranslation(Matrix &result,
                                      UInt32  OSG_CHECK_ARG(width ),
                                      UInt32  OSG_CHECK_ARG(height), int eye)
{
    result.setIdentity();
}

/*! Get/calculate the viewing matrix for this camera. This is the inverse
    of the beacon's toWorld transformation.
*/
void Camera::getViewing(Matrix &result,
                        UInt32  OSG_CHECK_ARG(width ),
                        UInt32  OSG_CHECK_ARG(height), int eye)
{
    if (getBeacon() == NullFC)
    {
        SWARNING << "Camera::setup: no beacon!" << std::endl;
        return;
    }

    getBeacon()->getToWorld(result);
    result.invert();
}

/*! Calculate the frustum of this camera's visible area.
*/
void Camera::getFrustum(FrustumVolume& result, const Viewport& p)
{
    Matrix mv,prt,pr;

    getProjection           (pr , p.getPixelWidth(), p.getPixelHeight());
    getProjectionTranslation(prt, p.getPixelWidth(), p.getPixelHeight());
    getViewing              (mv , p.getPixelWidth(), p.getPixelHeight());

    pr.mult(prt);
    pr.mult(mv );

    result.setPlanes(pr);
}

/*! Calculate the frustum of this camera's visible area (w,h instead port).
*/
void Camera::getFrustum(FrustumVolume& result,
                        UInt32  width, UInt32  height)
{
    Matrix mv,prt,pr;

    getProjection           (pr , width, height);
    getProjectionTranslation(prt, width, height);
    getViewing              (mv , width, height);

    pr.mult(prt);
    pr.mult(mv );

    result.setPlanes(pr);
}

/*! Calculate the matrix that transforms world coordinates into the screen
    coordinate system for this camera.
*/
void Camera::getWorldToScreen(Matrix &result, const Viewport& p)
{
    Matrix mv,prt,pr;

    getProjection           (result, p.getPixelWidth(), p.getPixelHeight());
    getProjectionTranslation(prt   , p.getPixelWidth(), p.getPixelHeight());
    getViewing              (mv    , p.getPixelWidth(), p.getPixelHeight());

    result.mult(prt);
    result.mult(mv );
}

/*! Get/calculate the decoration matrix for this camera. The default is identity.
*/
void Camera::getDecoration(Matrix &result, UInt32 width, UInt32 height)
{
    result.setIdentity();
}

bool Camera::setTransform(const Matrix &m, const CameraSpace space)
{
    Matrix mLocal = m;

    if (space != kLocalSpace)
        mLocal = convertMatToLocalSpace(m);

    // respect possible tracking matrix
    Matrix tm = _sfTrackingMatrix.getValue();
    tm.invert();
    mLocal.mult(tm);

    Vec3f trans, scale;
    Quaternion rot, so;
    mLocal.getTransform(trans, rot, scale, so);

    CameraPtr temp(*this);

    Matrix &current = temp->getMatrix();

    beginEditCP(temp, Camera::TranslationFieldMask | Camera::ScaleFieldMask);
        temp->setTranslation(trans);
        temp->setScale(scale);
    endEditCP(temp, Camera::TranslationFieldMask | Camera::ScaleFieldMask);

    // respect existing pivot translations
    Pnt3f posBefore(current[3][0], current[3][1], current[3][2]);

    temp->setRotationAsQuaternion(rot);

    Pnt3f posAfter(current[3][0], current[3][1], current[3][2]);

    Vec3f translationDiff = posBefore - posAfter;

    beginEditCP(temp, Camera::RotatePivotTranslationFieldMask);
        temp->setRotatePivotTranslation(temp->getRotatePivotTranslation() + translationDiff);
    endEditCP(temp, Camera::RotatePivotTranslationFieldMask);

    return (trans != temp->getTranslation() ||
            scale != temp->getScale() ||
            rot != temp->getRotation());
}

void Camera::getTransform(Matrix &m, const CameraSpace space) const
{
    if (space == Camera::kLocalSpace)
    {
        m = getMatrix();
        // respect tracking matrix for local space
        m.mult(_sfTrackingMatrix.getValue());
    }
    else
    {
        NodePtr beacon = getBeacon();

        if (beacon != NullFC)
            beacon->getToWorld(m);
        else
            SWARNING << "Camera::getTransform: no beacon!" << std::endl;
    }
}

Matrix Camera::getTransform(const CameraSpace space) const
{
    Matrix m;
    getTransform(m, space);
    return m;
}

void Camera::getParentTransform(Matrix &m, const CameraSpace space) const
{
    NodePtr beacon = getBeacon();

    if (beacon != NullFC)
    {
        NodePtr parent = beacon->getParent();

        if (parent != NullFC)
        {
            parent->getToWorld(m);

            if (space == Camera::kLocalSpace)
            {
                if (parent->getCore() != NullFC &&
                    parent->getCore()->getType().isDerivedFrom(Transform::getClassType()))
                {
                    TransformPtr trans = TransformPtr::dcast(parent->getCore());
                    m = trans->getMatrix();
                }
                else
                {
                    parent->getToWorld(m);

                    NodePtr parentParent = parent->getParent();
                    if (parentParent != NullFC)
                    {
                        Matrix pm;
                        parentParent->getToWorld(pm);
                        pm.invert();

                        m.multLeft(pm);
                    }
                }
            }
        }
        else
        {
            SWARNING << "Camera::getParentTransform: beacon has no parent!" << std::endl;
        }
    }
    else
    {
        SWARNING << "Camera::getParentTransform: no beacon!" << std::endl;
    }
}

Matrix Camera::getParentTransform(const CameraSpace space) const
{
    Matrix m;
    getParentTransform(m, space);
    return m;
}

Matrix Camera::convertMatToLocalSpace(const Matrix &mGlobal)
{
    Matrix mLocal = mGlobal;

    Matrix m;
    getParentTransform(m, Camera::kGlobalSpace);

    if (!m.equals(Matrix::identity(), Eps))
    {
        m.invert();
        mLocal.multLeft(m);
    }

    return mLocal;
}

Pnt3f Camera::convertPntToLocalSpace(const Pnt3f &pGlobal)
{
    Pnt3f pLocal = pGlobal;

    Matrix m;
    getParentTransform(m, Camera::kGlobalSpace);

    if (!m.equals(Matrix::identity(), Eps))
    {
        m.invert();
        m.multFullMatrixPnt(pLocal, pLocal);
    }

    return pLocal;
}

Vec3f Camera::convertVecToLocalSpace(const Vec3f &vGlobal)
{
    Vec3f vLocal = vGlobal;

    Matrix m;
    getParentTransform(m, Camera::kGlobalSpace);

    if (!m.equals(Matrix::identity(), Eps))
    {
        m.invert();
        m.multMatrixVec(vLocal, vLocal);
    }

    return vLocal;
}



Pnt3f Camera::convertPntToGlobalSpace(const Pnt3f &pLocal) const
{
    Pnt3f pGlobal = pLocal;

    Matrix m;
    getParentTransform(m, Camera::kGlobalSpace);

    if(!m.equals(Matrix::identity(), Eps))
    {
        m.multFullMatrixPnt(pGlobal, pGlobal);
    }

    return pGlobal;
}

Vec3f Camera::convertVecToGlobalSpace(const Vec3f &vLocal) const
{
    Vec3f vGlocal = vLocal;

    Matrix m;
    getParentTransform(m, Camera::kGlobalSpace);

    if(!m.equals(Matrix::identity(), Eps))
    {
        m.multMatrixVec(vGlocal, vGlocal);
    }

    return vGlocal;
}

void Camera::setAtDistanceToPoint(const Pnt3f& worldPos)
{
    // implemented by UberCamera
}


/*! Calculate a ray that starts at the camera position and goes through the
pixel \a x, \a y in the viewport \a port. \a x and \a y are relative to the
parent window's upper left corner. \a t is the length of the viewing ray.
*/
bool Camera::calcViewRay(Line &line, Int32 x, Int32 y, const Viewport& port)
{
    if(port.getPixelWidth() <= 0 || port.getPixelHeight() <= 0)
    {
        return false;
    }

    Matrix proj, projtrans, view;

    getProjection(proj, port.getPixelWidth(), port.getPixelHeight());
    getProjectionTranslation(projtrans, port.getPixelWidth(),
                                port.getPixelHeight());
    getViewing(view, port.getPixelWidth(), port.getPixelHeight());

    Real32  rx = (x - port.getPixelLeft()) / Real32(port.getPixelWidth())
                    * 2.f - 1.f,
            ry = 1.f - ((y - (port.getParent()->getHeight() -
                              port.getPixelTop())
                        ) /
                        Real32(port.getPixelHeight())
                       ) * 2.f;

    Matrix fullProj = proj;
    fullProj.mult(projtrans);
    
    Matrix invProj;
    invProj.invertFrom(fullProj);

    Pnt3f from, at;
    invProj.multFull(Pnt3f(rx, ry, -1.0f), from);
    invProj.multFull(Pnt3f(rx, ry,  -0.75f), at  );
    
    Vec3f dir = at - from;
    dir.normalize();

    Matrix invView;
    invView.invertFrom(view);

    invView.multMatrixVec(dir);
    invView.multFull(from, from);

    line.setValue(from, dir);

    return true;
}

CameraPtr Camera::getNonDecoratorCamera(const CameraPtr &cam)
{
    CameraDecoratorPtr firstDeco = NullFC;
    return getNonDecoratorCamera(cam, firstDeco);
}

CameraPtr Camera::getNonDecoratorCamera(const CameraPtr &cam, FieldContainerPtr &firstDecorator)
{
    if( cam == NullFC )
        return NullFC;

    if( !cam->getType().isDerivedFrom(CameraDecorator::getClassType()) )
        return cam;

    CameraPtr decoratee = NullFC;
    CameraDecoratorPtr deco = CameraDecoratorPtr::dcast(cam);
    while( deco != NullFC )
    {
        firstDecorator = deco;
        decoratee = deco->getDecoratee();
        deco = CameraDecoratorPtr::dcast(decoratee);
    }

    return decoratee;
}

bool Camera::pushTrackingUserMatrix(const CameraPtr &cam)
{
    if (cam == NullFC)
        return false;

    CameraPtr decoratee = NullFC;
    CameraDecoratorPtr deco = CameraDecoratorPtr::dcast(cam);

    while (deco != NullFC)
    {
        ProjectionCameraDecoratorPtr projDeco = ProjectionCameraDecoratorPtr::dcast(deco);

        if (projDeco != NullFC)
        {
            const Matrix &userMatrix = projDeco->getUserMatrix();

            if (!projDeco->getTrackingMatrix().equals(userMatrix, Eps))
            {
                projDeco->setTrackingMatrix(projDeco->getUserMatrix());
                return true;
            }
        }

        decoratee = deco->getDecoratee();
        deco = CameraDecoratorPtr::dcast(decoratee);
    }

    return false;
}

/*------------------------------- dump ----------------------------------*/

void Camera::dump(      UInt32    OSG_CHECK_ARG(uiIndent),
                  const BitVector OSG_CHECK_ARG(bvFlags )) const
{
    SLOG << "Dump Camera NI" << std::endl;
}


/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp       [] = "@(#)$Id: FCTemplate_cpp.h,v 1.13 2002/06/01 10:37:25 vossg Exp $";
    static Char8 cvsid_hpp       [] = OSGCAMERA_HEADER_CVSID;
    static Char8 cvsid_inl       [] = OSGCAMERA_INLINE_CVSID;

    static Char8 cvsid_fields_hpp[] = OSGCAMERAFIELDS_HEADER_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif

