/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright(C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include <OSGConfig.h>

#include <OSGGL.h>
#include <OSGRemoteAspect.h>
#include <OSGFieldContainer.h>
#include <OSGNode.h>
#include <OSGAction.h>
#include <OSGDrawAction.h>
#include "OSGViewport.h"
#include "OSGCamera.h"
#include "OSGWindow.h"
#include "OSGImage.h"

#include "OSGImageForeground.h"

OSG_USING_NAMESPACE


/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::ImageForeground
    \ingroup GrpSystemWindowForegrounds
    
The ImageForeground is used to draw images on top of the viewport.  See \ref
PageSystemWindowForegroundImage for a description.

The images are stored in the _mfImages Field, the corresponding positions in
the _mfPositions Field.

*/

/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  private                                                                -
\*-------------------------------------------------------------------------*/

void ImageForeground::initMethod(void)
{
}

/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*------------- constructors & destructors --------------------------------*/

ImageForeground::ImageForeground(void) :
    Inherited(),
    m_vertexArrayId(0),
    m_shader(osg::NullFC),
    m_isDirty(true)
{
}

ImageForeground::ImageForeground(const ImageForeground &source) :
    Inherited(source),
    m_vertexArrayId(0),
    m_shader(osg::NullFC),
    m_isDirty(true)
{
}

ImageForeground::~ImageForeground(void)
{

}

void ImageForeground::onCreate(const ImageForeground *source)
{
    Inherited::onCreate(source);

    // ignore prototypes.
    if (GlobalSystemState == Startup)
        return;

}

void ImageForeground::onDestroy(void)
{
    Inherited::onDestroy();

    if (m_shader != NullFC)
        subRefCP(m_shader);

    // remove unused textures
    while (!m_textures.empty())
    {
        subRefCP(m_textures.back());
        m_textures.pop_back();
    }
   
    if (osgMakeCurrent())
    {
        if (m_vertexArrayId != 0)
        {
            glDeleteVertexArrays(1, &m_vertexArrayId);
            m_vertexArrayId = 0;
        }
    }
}

void ImageForeground::changed(BitVector whichField, UInt32 origin)
{
    if (whichField & PositionsFieldMask)
    {
        m_isDirty = true;
    }
    if (whichField & ImagesFieldMask)
    {
        m_isDirty = true;
    }
    Inherited::changed(whichField, origin);
}

/*------------------------------- dump ----------------------------------*/

void ImageForeground::dump(     UInt32    OSG_CHECK_ARG(uiIndent), 
                           const BitVector OSG_CHECK_ARG(bvFlags)) const
{
    SLOG << "Dump ImageForeground NI" << std::endl;
}

void ImageForeground::draw(DrawActionBase* action, Viewport *vp)
{
    if (getActive() == false)
        return;

    if (getMFPositions()->size() != getMFImages()->size())
        return;

    // check if the images are valid
    for (UInt32 i = 0; i < getMFImages()->size(); i++)
    {
        if (getImages(i) == NullFC)
            return;
    }

    if (m_isDirty)
    {
        for (UInt32 i = 0; i < getMFImages()->size(); i++)
        {
            ImagePtr img = getImages(i);
            if (img == NullFC)
                continue;

            TextureChunkPtr texture = NullFC;
            if (i < m_textures.size())
                texture = m_textures[i];
            else
            {
                texture = TextureChunk::create();
                addRefCP(texture);
                m_textures.push_back(texture);
            }

            if (texture->getImage() != img)
            {
                beginEditCP(texture, osg::FieldBits::AllFields);
                texture->setImage(img);
                texture->setMinFilter(GL_LINEAR);
                texture->setMagFilter(GL_LINEAR);
                texture->setWrapS(GL_CLAMP_TO_BORDER);
                texture->setWrapT(GL_CLAMP_TO_BORDER);
                endEditCP(texture, osg::FieldBits::AllFields);
            }
        }
        // remove unused textures
        while( m_textures.size() > getMFImages()->size())
        { 
            subRefCP(m_textures.back());
            m_textures.pop_back();
        }
    }
    if (m_shader == NullFC)
    {
        std::string vp = 
            "const vec2 quadVertices[4] = vec2[4](vec2(-1.0, -1.0), vec2(1.0, -1.0), vec2(-1.0, 1.0), vec2(1.0, 1.0)); \n"
            "const vec2 quadTexCoords[4] = vec2[4](vec2(0.0, 0.0), vec2(1.0, 0.0), vec2(0.0, 1.0), vec2(1.0, 1.0)); \n"
            "uniform vec4 textureRegion; \n"
            "out vec2 texCoord; \n"
            "void main(void) \n"
            "{\n"
            "   texCoord = (quadTexCoords[gl_VertexID].xy - textureRegion.xz) / textureRegion.yw; \n" // mix(textureRegion.xz, textureRegion.yw - textureRegion.xz, quadTexCoords[gl_VertexID].xy); \n"
            "   gl_Position = vec4(quadVertices[gl_VertexID], 0.0, 1.0); \n"
            "}\n";

        std::string fp = 
            "uniform sampler2D textureMap;\n"
            "uniform float alpha; \n"
            "in vec2 texCoord; \n"
            "layout (location = 0) out vec4 colorBufferRGBA;\n"
            "void main()  \n"
            "{  \n"
            "   vec4 color = texture(textureMap, texCoord); \n"
            "   colorBufferRGBA = vec4(color.rgb, color.a * alpha); \n"
            "} \n";

        m_shader = SHLChunk::create();
        beginEditCP(m_shader, osg::FieldBits::AllFields);
        m_shader->setSingleViewRenderingOnly(true);
        m_shader->setVertexProgram(vp);
        m_shader->setGeometryProgram("");
        m_shader->setFragmentProgram(fp);
        m_shader->clearUniformParameters();
        endEditCP(m_shader, osg::FieldBits::AllFields);
        addRefCP(m_shader);
    }

    
    

    // retrieve the vertex array
    if (m_vertexArrayId == 0)
        glGenVertexArrays(1, &m_vertexArrayId);

    pushGLFlags();
    disableGLFlag(GL_DEPTH_TEST);


    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    enableGLFlag(GL_BLEND);

    osg::Vec2f portSize = osg::Vec2f(static_cast<osg::Real32>(vp->getPixelWidth()), static_cast<osg::Real32>(vp->getPixelHeight()));
    float vpWidth = 1.0, vpHeight = 1.0;
    if (vp)
    {
        // for absolute pixel position
        vpWidth = 1.0 / portSize[0];
        vpHeight = 1.0 / portSize[1];
    }

    if (m_shader != NullFC)
    {
        glBindVertexArray(m_vertexArrayId);

        for (UInt32 i = 0; i < getMFPositions()->size(); i++)
        {
            Pnt2f p = getPositions(i);
            ImagePtr img = getImages(i);

            beginEditCP(m_shader, ShaderChunk::ParametersFieldMask);
            m_shader->setUniformParameter("textureMap", 0);
            m_shader->setUniformParameter("alpha", getAlpha());
            osg::Vec4f textureRegion;
            if (p[0] >= 1.0 || p[1] >= 1.0)
                textureRegion = osg::Vec4f(p[0] * vpWidth, img->getWidth() * vpWidth, p[1] * vpHeight, img->getHeight() * vpHeight);
            else
                textureRegion = osg::Vec4f(p[0], img->getWidth() * vpWidth, p[1], img->getHeight() * vpHeight);

            m_shader->setUniformParameter("textureRegion", textureRegion);
            endEditCP(m_shader, ShaderChunk::ParametersFieldMask);

            m_textures[i]->activate(action, 0);

            m_shader->activate(action);
            glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
            m_shader->deactivate(action);

            m_textures[i]->deactivate(action, 0);
           
        }
        glBindVertexArray(0);
    }

    popGLFlags();
#if 0
   

    UInt16 i;
    
    for(i = 0; i < getMFPositions()->size(); i++)
    {
        if(getImages(i) != NullFC)
            break;
    }
    
    if(i == getMFPositions()->size())   // all images == NULL?
        return; 

    pushGLFlags();
    disableGLFlag(GL_DEPTH_TEST);


    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    enableGLFlag(GL_BLEND);

    float vpWidth = 1.0, vpHeight = 1.0;
    if(vp)
    {
        // for absolute pixel position
        vpWidth  = 1.0/vp->getPixelWidth();
        vpHeight = 1.0/vp->getPixelHeight();
    }

    for(i = 0; i < getMFPositions()->size(); i++)
    {
        ImagePtr img = getImages(i);

        if(img == NullFC)
            continue;

        Pnt2f p = getPositions(i);
        if( p[0] >= 1.0 || p[1] >= 1.0 )
            glRasterPos2f(p[0]*vpWidth, p[1]*vpHeight); // absolute position
        else
            glRasterPos2f(p[0], p[1]); // relative position

        glDrawPixels(img->getWidth(), img->getHeight(),
                     img->getPixelFormat(), OSG_GL_UNSIGNED_BYTE,
                     img->getData());
    }
    popGLFlags();
#endif
}

/*-------------------------------------------------------------------------*/
/*                              cvs id's                                   */

#ifdef __sgi
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning(disable : 177)
#endif

namespace
{
    static char cvsid_cpp[] = "@(#)$Id: $";
    static char cvsid_hpp[] = OSGIMAGEFOREGROUND_HEADER_CVSID;
    static char cvsid_inl[] = OSGIMAGEFOREGROUND_INLINE_CVSID;
}


