/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright(C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "OSGConfig.h"

#include <OSGBaseTypes.h>
#include <OSGLog.h>
#include <OSGVector.h>
#include <OSGMatrix.h>
#include <OSGMatrixUtility.h>

#include "OSGNodePtr.h"
#include "OSGViewport.h"
#include "OSGCamera.h"
#include "OSGWindow.h"
#include "OSGBackground.h"
#include "OSGPerspectiveCamera.h"

OSG_USING_NAMESPACE


/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::PerspectiveCamera
    \ingroup GrpSystemWindowCameras

The Perspective Camera class, see \ref PageSystemWindowCameraPerspective for 
a description.

The only new parameter is the _sfFov.
*/

/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  private                                                                -
\*-------------------------------------------------------------------------*/

void PerspectiveCamera::initMethod(void)
{
}

/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
 -  public                                                                 -
\*-------------------------------------------------------------------------*/


/*------------- constructors & destructors --------------------------------*/

PerspectiveCamera::PerspectiveCamera(void) :
    Inherited()
{
}

PerspectiveCamera::PerspectiveCamera(const PerspectiveCamera &source) :
    Inherited(source)
{
}

PerspectiveCamera::~PerspectiveCamera(void)
{
}

void PerspectiveCamera::changed(BitVector whichField, UInt32 origin)
{
    Inherited::changed(whichField, origin);
}

/*-------------------------- your_category---------------------------------*/

void PerspectiveCamera::draw(      DrawAction *OSG_CHECK_ARG(action), 
                             const Viewport&   OSG_CHECK_ARG(port ))
{
}

void PerspectiveCamera::getProjection(Matrix& result,
    UInt32 width, UInt32 height, int eye, const bool reverseZ)
{
    Real32 fov = getFov();

    // catch some illegal cases
    if(fov < 0 || width == 0 || height == 0)
    {
        result.setIdentity();
        return;
    }

    // try to be nice to people giving degrees...
    if(fov > Pi)
        fov = osgdegree2rad(fov);

    Real32 rNear = getNear(), rFar = getFar();
    Real32 aspect = Real32(width) / Real32(height) * getAspect();
    Real32 ct = osgtan(fov / 2);

    if(rNear > rFar)
    {
        SWARNING << "MatrixPerspective: near " << rNear << " > far " << rFar
                 << "!\n" << std::endl;
        result.setIdentity();
        return;
    }

    if(fov <= Eps)
    {
        SWARNING << "MatrixPerspective: fov " << fov << " very small!\n"
                 << std::endl;
        result.setIdentity();
        return;
    }

    if(osgabs(rNear - rFar) < Eps)
    {
        SWARNING << "MatrixPerspective: near " << rNear << " ~= far " << rFar
                 << "!\n" << std::endl;
        result.setIdentity();
        return;
    }

    if(aspect < Eps)
    {
        SWARNING << "MatrixPerspective: aspect ratio " << aspect
                 << " very small!\n" << std::endl;
        result.setIdentity();
        return;
    }

    Real32 x = ct * rNear, y = ct * rNear;
    UInt32 fovMode = getFovMode();
    switch (fovMode)
    {
    case FOV_vertical:
        x *= aspect;
        break;
    case FOV_horizontal:
        y /= aspect;
        break;
    case FOV_smaller:
        if (width * getAspect() >= height)
            x *= aspect;
        else
            y /= aspect;
        break;
    default:
        result.setIdentity();
        return;
    }
    // TODO REVERSE Z
    MatrixFrustum(result,
                  -x,
                   x,
                  -y,
                   y,
                   rNear,
                   rFar,
                   reverseZ);
}

bool PerspectiveCamera::hasProjectionOverride()
{
    return true;
}


/*------------------------------- dump ----------------------------------*/

void PerspectiveCamera::dump(      UInt32    OSG_CHECK_ARG(uiIndent), 
                             const BitVector OSG_CHECK_ARG(bvFlags)) const
{
    SLOG << "Dump PerspectiveCamera NI" << std::endl;
}



/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning(disable : 177)
#endif

namespace
{
    static Char8 cvsid_cpp       [] = "@(#)$Id: FCTemplate_cpp.h,v 1.13 2002/06/01 10:37:25 vossg Exp $";
    static Char8 cvsid_hpp       [] = OSGPERSPECTIVECAMERA_HEADER_CVSID;
    static Char8 cvsid_inl       [] = OSGPERSPECTIVECAMERA_INLINE_CVSID;

    static Char8 cvsid_fields_hpp[] = OSGPERSPECTIVECAMERAFIELDS_HEADER_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif
