/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000-2002 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------


#include <stdlib.h>
#include <stdio.h>

#include <OSGConfig.h>

#include "OSGNodePtr.h"
#include "OSGShadowCameraDecorator.h"
//#include <OSGPerspectiveCamera.h>

OSG_USING_NAMESPACE


/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::OSGShadowCameraDecorator
    \ingroup GrpSystemWindowCameraDecorators

The osg::OSGShadowCameraDecorator is to be used exclusively for shadow mapping.
It adds an offset the the viewing matrix of the camera and removes any scaling.
It will not call getViewing of the decoratee but calculates the matrix
manually from the beacon, removes scaling, offsets it and inverts it.
*/

/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

void ShadowCameraDecorator::initMethod (void)
{
}

/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*------------- constructors & destructors --------------------------------*/

ShadowCameraDecorator::ShadowCameraDecorator(void)
    : Inherited()
{
}

ShadowCameraDecorator::ShadowCameraDecorator(const ShadowCameraDecorator &source)
    : Inherited(source)
{
}

ShadowCameraDecorator::~ShadowCameraDecorator(void)
{
}

void ShadowCameraDecorator::changed(BitVector whichField, UInt32 origin)
{
    Inherited::changed(whichField, origin);
}

void ShadowCameraDecorator::getViewing(Matrix &result, UInt32 width, UInt32 height, int eye)
{
    CameraPtr camera = getDecoratee();
    if (camera == NullFC)
    {
        FWARNING(("OSGShadowCameraDecorator::getViewing: no decoratee!\n"));
        result.setIdentity();
        return;
    }

    const Real32 offset = getOffset();
    if (offset == 0.0f)
    {
        camera->getViewing(result, width, height, eye);
        return;
    }

    // Note: Offset > 0 is set for planar area lights (disk & rectangular light).
    // With the offset the shadowmap is rendered from a point behind the actual light.
    // The near clipping plane should be set to offset to coincide with the light plane.

    // Get the light-to-world transform.
    camera->getBeacon()->getToWorld(result);
    Vec3f translate, scale;
    Quaternion rotate, scaleOrientation;
    result.getTransform(translate, rotate, scale, scaleOrientation);
    
    // Calculate the offset vector in world space
    Vec3f backward(0.0f, 0.0f, 1.0f);
    Matrix transposedInverse;
    transposedInverse.invertFrom3(result);
    transposedInverse.transpose();
    transposedInverse.mult(backward);
    backward.normalize();

    // - Remove scaling to make better use the shadowmap resolution.
    // - Ensure matrix is right-handed.
    //   Otherwise view matrix of shadow map camera would switch orientation of faces

    const bool handednessChanged = result[0].cross(result[1]).dot(result[2]) < 0.0f; 
    Matrix invScale;
    invScale.setScale((handednessChanged ? -1.0f : 1.0f) / fabs(scale[0]), 1.0f / fabs(scale[1]), 1.0f / fabs(scale[2]));
    result.mult(invScale);

    // Apply the offset to the matrix.
    result.setTranslate(translate + backward * offset);

    // View matrix is world-to-light transform
    result.invert();
}
/*------------------------------- dump ----------------------------------*/

void ShadowCameraDecorator::dump(UInt32 OSG_CHECK_ARG(uiIndent), 
                               const BitVector OSG_CHECK_ARG(bvFlags )) const
{
    SLOG << "Dump ShadowCameraDecorator NI" << std::endl;
}

/*-------------------------------------------------------------------------*/
/*                              cvs id's                                   */

#ifdef __sgi
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static char cvsid_cpp[] = "@(#)$Id: $";
    static char cvsid_hpp[] = OSGShadowCameraDecorator_HEADER_CVSID;
    static char cvsid_inl[] = OSGSHADOWCAMERADECORATOR_INLINE_CVSID;
    }
