/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *               Copyright (C) 2000-2002 by the OpenSG Forum                 *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
 \*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
 \*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 \*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>

#include <OSGConfig.h>
#include <OSGGL.h>
#include <OSGDrawActionBase.h>
#include <OSGViewport.h>
#include <OSGTextureChunk.h>

#include "OSGTextureBackground.h"

#include<iostream>

OSG_USING_NAMESPACE

/***************************************************************************\
 *                            Description                                  *
\***************************************************************************/

/*! \class osg::TextureBackground
  A textured background e.g for distortion corrected fast live video from a camera with known intrinsic parameters.
*/

/***************************************************************************\
 *                           Class variables                               *
\***************************************************************************/

/***************************************************************************\
 *                           Class methods                                 *
\***************************************************************************/

void TextureBackground::initMethod (void)
{
}


/***************************************************************************\
 *                           Instance methods                              *
\***************************************************************************/

/*-------------------------------------------------------------------------*\
  -  private                                                                 -
  \*-------------------------------------------------------------------------*/

/*----------------------- constructors & destructors ----------------------*/

TextureBackground::TextureBackground(void) :
    Inherited()
{
}

TextureBackground::TextureBackground(const TextureBackground &source) :
    Inherited(source)
{
}

TextureBackground::~TextureBackground(void)
{
}

/*----------------------------- class specific ----------------------------*/




void TextureBackground::changed(BitVector whichField, UInt32 origin)
{
    Inherited::changed(whichField, origin);
 
    // all updates are handled in updateGrid()
}


void TextureBackground::updateGrid(void)
{
    bool gridChanged=( (getHor() != _hor) || 
                       (getVert() != _vert) );
                       
    if(gridChanged)
    {
        //resize grid
	    UInt32 gridCoords=(getHor()+2)*(getVert()+2);
	    _textureCoordArray.resize(gridCoords);
	    _vertexCoordArray.resize(gridCoords);
        
	    int indexArraySize=(getHor()+2)*((getVert()+1)*2);
	    _indexArray.resize(indexArraySize);
        
        _hor = getHor();
        _vert = getVert();
    }
    
    if(gridChanged || _radialDistortion   != getRadialDistortion() ||
                      _centerOfDistortion != getCenterOfDistortion()
      )
    {
        _radialDistortion = getRadialDistortion();
        _centerOfDistortion = getCenterOfDistortion();
        
	    // calculate grid coordinates and triangle strip indices
	    float xStep=1.0/float(getHor()+1);
	    float yStep=1.0/float(getVert()+1);
	    std::vector<Vec2f>::iterator texCoord=_textureCoordArray.begin();
	    std::vector<Vec2f>::iterator vertexCoord=_vertexCoordArray.begin();
	    std::vector<UInt32>::iterator index=_indexArray.begin();
	    UInt32 coord0(0),coord1(0);
	    OSGGLfloat x,y;
	    Int16 xx,yy;
	    Int16 xxmax=getHor()+2,yymax=getVert()+2;
	    for(yy=0,y=0.0f;yy<yymax;yy++,y+=yStep)
	    {
	        if(yy>0)
	        {
		        coord1=yy*xxmax;
		        coord0=coord1-xxmax;
		        *index++=coord1++;
		        *index++=coord0++;
	        }
	        float dy=y-getCenterOfDistortion().y();
	        float dy2=dy*dy;
	        for(xx=0,x=0.0f;xx<xxmax;xx++,x+=xStep)
	        {
		        *texCoord++=Vec2f(x,y);
		        float dx=(x-getCenterOfDistortion().x());
		        float dx2=dx*dx;
		        float dist2=dx2+dy2;
		        float deltaX=dx*getRadialDistortion()*dist2;
		        float deltaY=dy*getRadialDistortion()*dist2;
		        *vertexCoord++=Vec2f(x+deltaX,y+deltaY);
		        if(yy>0&&xx>0)
		        {
		            *index++=coord1++;
		            *index++=coord0++;
		        }
	        }
	    }
    }
}

void TextureBackground::clear(DrawActionBase *action, Viewport *OSG_CHECK_ARG(viewport))
{
    FFATAL(("TextureBackground no longer supported in core profile!\n"));
}


void TextureBackground::dump(      UInt32    ,
				   const BitVector ) const
{
    SLOG << "Dump TextureBackground NI" << std::endl;
}


/*------------------------------------------------------------------------*/
/*                              cvs id's                                  */

#ifdef OSG_SGI_CC
#pragma set woff 1174
#endif

#ifdef OSG_LINUX_ICC
#pragma warning( disable : 177 )
#endif

namespace
{
    static Char8 cvsid_cpp       [] = "@(#)$Id: OSGTextureBackground.cpp,v 1.9 2008/06/09 07:30:42 vossg Exp $";
    static Char8 cvsid_hpp       [] = OSGTEXTUREBACKGROUNDBASE_HEADER_CVSID;
    static Char8 cvsid_inl       [] = OSGTEXTUREBACKGROUNDBASE_INLINE_CVSID;

    static Char8 cvsid_fields_hpp[] = OSGTEXTUREBACKGROUNDFIELDS_HEADER_CVSID;
}

#ifdef __sgi
#pragma reset woff 1174
#endif

