#include "FieldContainerViewCtl.h"

#include <cstdlib>

#include <iostream>

#include <QComboBox>
#include <QFileDialog>
#include <QMessageBox>
#include <QTextEdit>
#include <QCheckBox>
#include <QButtonGroup>
#include <QLineEdit>
#include <QListView>
#include <QDir>

/* 
 *  Constructs a FieldContainerViewCtl which is a child of 'parent', with the 
 *  name 'name' and widget flags set to 'f' 
 *
 *  The dialog will by default be modeless, unless you set 'modal' to
 *  TRUE to construct a modal dialog.
 */
FieldContainerViewCtl::FieldContainerViewCtl( const char *file, QWidget *parent, const QString &name )
    : QDialog(parent),
      _activePart(0), _activePartIndex(0),
	  _lineUpdate(false),
	  _procGroup(0)
{
    setupUi(this);

    QObject::connect(partListBox, &QListWidget::clicked, this, &FieldContainerViewCtl::activePartChanged);
    QObject::connect(load, &QPushButton::clicked, this, &FieldContainerViewCtl::loadSlot);
    QObject::connect(save, &QPushButton::clicked, this, &FieldContainerViewCtl::saveSlot);
    QObject::connect(newPart, &QPushButton::clicked, this, &FieldContainerViewCtl::newPartSlot);
    QObject::connect(deletePart, &QPushButton::clicked, this, &FieldContainerViewCtl::deletePartSlot);
    QObject::connect(clonePart, &QPushButton::clicked, this, &FieldContainerViewCtl::clonePartSlot);
    QObject::connect(nodeNameInput, &QLineEdit::textChanged, this, &FieldContainerViewCtl::nameChanged);
    QObject::connect(nodeParentInput, &QLineEdit::textChanged, this, &FieldContainerViewCtl::parentChanged);
    QObject::connect(partNameInput, &QLineEdit::textChanged, this, &FieldContainerViewCtl::partNameChanged);
    QObject::connect(partDefaultValueInput, &QLineEdit::textChanged, this, &FieldContainerViewCtl::partDefaultValueChanged);
    QObject::connect(partDescriptionInput, &QLineEdit::textChanged, this, &FieldContainerViewCtl::partDescriptionChanged);
    QObject::connect(partVisibilityCombo, &QComboBox::activated, this, &FieldContainerViewCtl::partVisibilityChanged);
    QObject::connect(loadParent, &QPushButton::clicked, this, &FieldContainerViewCtl::loadParentSlot);
    QObject::connect(saveAs, &QPushButton::clicked, this, &FieldContainerViewCtl::saveAsSlot);
    QObject::connect(exit, &QPushButton::clicked, this, &FieldContainerViewCtl::exitSlot);
    QObject::connect(upPart, &QPushButton::clicked, this, &FieldContainerViewCtl::upPartSlot);
    QObject::connect(downPart, &QPushButton::clicked, this, &FieldContainerViewCtl::downPartSlot);
    QObject::connect(about, &QPushButton::clicked, this, &FieldContainerViewCtl::aboutSlot);
    QObject::connect(writeNodeCode, &QPushButton::clicked, this, &FieldContainerViewCtl::writeFieldContainerBaseSlot);
    QObject::connect(pointerFieldTypesCombo, &QComboBox::activated, this, &FieldContainerViewCtl::pointerFieldTypesChanged);
    QObject::connect(partCardinalityCombo, &QComboBox::activated, this, &FieldContainerViewCtl::partCardinalityChanged);
    QObject::connect(nodeLibraryInput, &QLineEdit::textChanged, this, &FieldContainerViewCtl::libraryChanged);
    QObject::connect(structureCombo, &QComboBox::activated, this, &FieldContainerViewCtl::structureChanged);
    QObject::connect(partAccessCombo, &QComboBox::activated, this, &FieldContainerViewCtl::partAccessChanged);
    QObject::connect(partIncludeInput, &QLineEdit::textChanged, this, &FieldContainerViewCtl::partIncludeChanged);
    QObject::connect(partDefaultHeaderInput, &QLineEdit::textChanged, this, &FieldContainerViewCtl::partDefaultHeaderChanged);
    QObject::connect(sysComp, &QCheckBox::stateChanged, this, &FieldContainerViewCtl::sysCompChanged);
    QObject::connect(parentSysComp, &QCheckBox::stateChanged, this, &FieldContainerViewCtl::parentSysCompChanged);
    QObject::connect(partTypeCombo, &QComboBox::currentTextChanged, this, &FieldContainerViewCtl::partTypeChanged);
    QObject::connect(decoratableSwitch, &QCheckBox::stateChanged, this, &FieldContainerViewCtl::decoratableSwitch_stateChanged);
    QObject::connect(writeProcCode, &QPushButton::clicked, this, &FieldContainerViewCtl::writeFieldContainerSlot);
    QObject::connect(localIncludes, &QCheckBox::stateChanged, this, &FieldContainerViewCtl::localIncludesChanged);
    QObject::connect(newButton, &QPushButton::clicked, this, &FieldContainerViewCtl::newSlot);
    QObject::connect(resetFieldTypes, &QPushButton::clicked, this, &FieldContainerViewCtl::resetFieldTypesSlot);
    QObject::connect(loadFieldTypes, &QPushButton::clicked, this, &FieldContainerViewCtl::loadFieldTypesSlot);

	int i;

	for (i = 0; true; i++) 
		if (FieldContainer::pointerFieldTypesStr(i))
        {
            QString pointFieldTypeStr = QString(FieldContainer::pointerFieldTypesStr(i));
			pointerFieldTypesCombo->addItem(pointFieldTypeStr);
        }
		else
			break;

	for (i = 0; true; i++) 
		if (FieldContainer::abstractStr(i))
			structureCombo->addItem(QString(FieldContainer::abstractStr(i)));
		else
			break;

	
	for (i = 0; true; i++) 
		if (Field::typeStr(i))
			partTypeCombo->addItem(QString(Field::typeStr(i)));
		else
			break;

	for (i = 0; true; i++)
		if (Field::cardinalityStr(i))
			partCardinalityCombo->addItem(QString(Field::cardinalityStr(i)));
		else
			break;

	for (i = 0; true; i++)
		if (Field::visibilityStr(i))
			partVisibilityCombo->addItem(QString(Field::visibilityStr(i)));
		else
			break;

	for (i = 0; true; i++)
		if (Field::accessStr(i))
			partAccessCombo->addItem(QString(Field::accessStr(i)));
		else
			break;

	if (file && *file)
		loadFieldContainer(file);
	
	if (_fieldContainer.numOfFields() == 0)
		partFrame->setEnabled(false);	    
}

/*  
 *  Destroys the object and frees any allocated resources
 */
FieldContainerViewCtl::~FieldContainerViewCtl()
{
    // no need to delete child widgets, Qt does it all for us
}

void FieldContainerViewCtl::sync(void)
{
	QString str = nodeDescriptionInput->toPlainText().simplified();

	_fieldContainer.setDescription(str.toUtf8().constData());
}

void FieldContainerViewCtl::updateList(void) 
{
	char line[1024];
	std::list<Field>::iterator nPI;

	partListBox->clear();

	for ( nPI = _fieldContainer.fieldList().begin();
				nPI != _fieldContainer.fieldList().end(); ++nPI ) {
		nPI->getLine(line);
		partListBox->addItem (QString(line));
	}
}

void FieldContainerViewCtl::updateActiveListItem(void) 
{
	char line[1024];

	_activePart->getLine(line);
	_lineUpdate = true;
    partListBox->item(_activePartIndex)->setText(QString(line));
	_lineUpdate = false;
}

void FieldContainerViewCtl::updateFieldContainerView(void)
{
    // without the block signals the slots are called and it crashes!
    nodeNameInput->blockSignals(true);
    nodeNameInput->setText(QString(_fieldContainer.name()));
    nodeNameInput->blockSignals(false);

    nodeParentInput->blockSignals(true);
    nodeParentInput->setText(QString(_fieldContainer.parentFieldContainer()));
    nodeParentInput->blockSignals(false);
    
    nodeLibraryInput->blockSignals(true);
    nodeLibraryInput->setText(QString(_fieldContainer.library()));
    nodeLibraryInput->blockSignals(false);
    
    nodeDescriptionInput->blockSignals(true);
    nodeDescriptionInput->setText(QString(_fieldContainer.description()));
    nodeDescriptionInput->blockSignals(false);
    
    pointerFieldTypesCombo->blockSignals(true);
    pointerFieldTypesCombo->setCurrentIndex(_fieldContainer.pointerFieldTypes());
    pointerFieldTypesCombo->blockSignals(false);
    
    structureCombo->blockSignals(true);
    structureCombo->setCurrentIndex(_fieldContainer.abstract());
    structureCombo->blockSignals(false);
    
    sysComp->blockSignals(true);
    sysComp->setChecked(_fieldContainer.systemComponent());
    sysComp->blockSignals(false);
    
    parentSysComp->blockSignals(true);
    parentSysComp->setChecked(_fieldContainer.parentSystemComponent());
    parentSysComp->blockSignals(false);
    
    decoratableSwitch->blockSignals(true);
    decoratableSwitch->setChecked(_fieldContainer.decoratable());
    decoratableSwitch->blockSignals(false);
    
    localIncludes->blockSignals(true);
    localIncludes->setChecked(_fieldContainer.useLocalIncludes());
    localIncludes->blockSignals(false);
}

void FieldContainerViewCtl::updateActivePartView(void)
{
    // without the block signals the slots are called and it crashes!
    if (_activePart) {
        partNameInput->blockSignals(true);
        partNameInput->setText(QString(_activePart->name()));
        partNameInput->blockSignals(false);
        
        partTypeCombo->blockSignals(true);
        partTypeCombo->setEditText(_activePart->type());
        partTypeCombo->blockSignals(false);

        partVisibilityCombo->blockSignals(true);
        partVisibilityCombo->setCurrentIndex(int(_activePart->visibility()));
        partVisibilityCombo->blockSignals(false);
        
        partCardinalityCombo->blockSignals(true);
        partCardinalityCombo->setCurrentIndex(int(_activePart->cardinality()));
        partCardinalityCombo->blockSignals(false);
        
        partAccessCombo->blockSignals(true);
        partAccessCombo->setCurrentIndex(int(_activePart->access()));
        partAccessCombo->blockSignals(false);
        
        partIncludeInput->blockSignals(true);
        partIncludeInput->setText(QString(_activePart->header()));
        partIncludeInput->blockSignals(false);
        
        partDefaultValueInput->blockSignals(true);
        partDefaultValueInput->setText(QString(_activePart->defaultValue()));
        partDefaultValueInput->blockSignals(false);
        
        partDefaultHeaderInput->blockSignals(true);
        partDefaultHeaderInput->setText(QString(_activePart->defaultHeader()));
        partDefaultHeaderInput->blockSignals(false);
        
        partDescriptionInput->blockSignals(true);
        partDescriptionInput->setText(QString(_activePart->description()));
        partDescriptionInput->blockSignals(false);
    }
    else {
        partNameInput->blockSignals(true);
        partNameInput->clear();
        partNameInput->blockSignals(false);
        
        partTypeCombo->blockSignals(true);
        partTypeCombo->setCurrentIndex(0);
        partTypeCombo->blockSignals(false);

        partVisibilityCombo->blockSignals(true);
        partVisibilityCombo->setCurrentIndex(0);
        partVisibilityCombo->blockSignals(false);
        
        partCardinalityCombo->blockSignals(true);
        partCardinalityCombo->setCurrentIndex(0);
        partCardinalityCombo->blockSignals(false);
        
        partAccessCombo->blockSignals(true);
        partAccessCombo->setCurrentIndex(0);
        partAccessCombo->blockSignals(false);
        
        partIncludeInput->blockSignals(true);
        partIncludeInput->clear();
        partIncludeInput->blockSignals(false);
        
        partDefaultValueInput->blockSignals(true);
        partDefaultValueInput->clear();
        partDefaultValueInput->blockSignals(false);
        
        partDescriptionInput->blockSignals(true);
        partDescriptionInput->clear();
        partDescriptionInput->blockSignals(false);
    }

}

void FieldContainerViewCtl::loadFieldContainer(const char *fileName)
{
	int i;

	if (fileName) {
		_fieldContainer.readDesc(fileName);
        setWindowTitle( tr( "OpenSG Field Container Editor : %1 ").arg(fileName) );
        _activePart = NULL;
        _activePartIndex = -1;
                
		updateFieldContainerView();
		if (_fieldContainer.fieldList().size()) {
			partTypeCombo->clear();
			for (i = 0; true; i++) 
				if (Field::typeStr(i))
					partTypeCombo->addItem(QString(Field::typeStr(i)));
				else
					break;
			partFrame->setEnabled(true);
			updateList();
			_activePartIndex = 0;
			_activePart = _fieldContainer.getField(_activePartIndex);
			updateActivePartView();
		}
		else {
			partFrame->setEnabled(false);
			_activePartIndex = -1;
			_activePart = 0;
			updateList();
			updateActivePartView();
		}
	}
}

/* 
 * public slot
 */
void FieldContainerViewCtl::activePartChanged(const QModelIndex & modelIndex)
{
	//	std::cerr << "activePartChanged:" << index << std::endl;

	if (!_lineUpdate) {
		_activePartIndex = modelIndex.row();
		_activePart = _fieldContainer.getField(_activePartIndex);
		if (_activePart) 
			updateActivePartView();
	}
}
/* 
 * public slot
 */
void FieldContainerViewCtl::clonePartSlot()
{
	int newPart;

	if (_activePart) {
        _activePartIndex = _fieldContainer.newField(*_activePart);
		_activePart = _fieldContainer.getField(_activePartIndex);
        
		partListBox->clearSelection();
		
		updateActivePartView();
		
		updateList();

		partListBox->setCurrentRow(_activePartIndex);
        
	}
}
/* 
 * public slot
 */
void FieldContainerViewCtl::deletePartSlot()
{
	int partNum;

	if (_activePart) {
		_fieldContainer.delField(_activePartIndex);
		partNum = _fieldContainer.numOfFields();
		if (partNum) {
			_activePartIndex = partNum - 1;
			_activePart = _fieldContainer.getField(_activePartIndex);
            partListBox->clearSelection();
			updateActivePartView();			
			updateList();
            partListBox->setCurrentRow(_activePartIndex);			
		}
		else {
			partFrame->setEnabled(false);
			_activePartIndex = -1;
			_activePart = 0;
			updateActivePartView();			
			updateList();
		}
	}
}

/* 
 * public slot
 */
void FieldContainerViewCtl::exitSlot()
{
	::exit(0);
}

/* 
 * public slot
 */
void FieldContainerViewCtl::loadSlot()
{
	QString fN (QFileDialog::getOpenFileName ( this, "FieldContainer Description File (*.fcd)" ) );

	if (!fN.isEmpty()) 
    {
		loadFieldContainer(fN.toUtf8().constData());
	}
	
}

/* 
 * public slot
 */
void FieldContainerViewCtl::nameChanged(const QString &value)
{
	_fieldContainer.setName(value.toUtf8().constData());
}

/* 
 * public slot
 */
void FieldContainerViewCtl::libraryChanged(const QString &value)
{
	_fieldContainer.setLibrary(value.toUtf8().constData());
}

/* 
 * public slot
 */
void FieldContainerViewCtl::newPartSlot()
{
	Field part;

	_activePartIndex = _fieldContainer.newField(part);
	_activePart = _fieldContainer.getField(_activePartIndex);

    partListBox->clearSelection();

	partFrame->setEnabled(true);

	partNameInput->setFocus();

	updateActivePartView();

	updateList();

    partListBox->setCurrentRow(_activePartIndex);
}

void FieldContainerViewCtl::upPartSlot()
{
	int newPos;

	if (_activePart && _fieldContainer.moveFieldLeft(_activePartIndex)) {
            _activePartIndex = _activePartIndex - 1;
			updateList();
            partListBox->setCurrentRow(_activePartIndex);
	}
}

void FieldContainerViewCtl::downPartSlot()
{
	int newPos;

	if (_activePart && _fieldContainer.moveFieldRight(_activePartIndex)) {
            _activePartIndex = _activePartIndex + 1;
			updateList();
            partListBox->setCurrentRow(_activePartIndex);            
	}
}

/* 
 * public slot
 */
void FieldContainerViewCtl::parentChanged(const QString &value)
{
	_fieldContainer.setParentFieldContainer(value.toUtf8().constData());
}

/* 
 * public slot
 */
void FieldContainerViewCtl::partDefaultValueChanged(const QString &value)
{
	if (_activePart) {
		_activePart->setDefaultValue(value.toUtf8().constData());
		updateActiveListItem();
	}
}
/* 
 * public slot
 */
void FieldContainerViewCtl::partDescriptionChanged(const QString &value)
{
	if (_activePart) {
		_activePart->setDescription(value.toUtf8().constData());
		updateActiveListItem();
	}
}
/* 
 * public slot
 */
void FieldContainerViewCtl::partNameChanged(const QString &value)
{
	if (_activePart) {
		_activePart->setName(value.toUtf8().constData());
		updateActiveListItem();
	}
}
/* 
 * public slot
 */
void FieldContainerViewCtl::partTypeChanged(const QString &value)
{
	if (_activePart) {
		_activePart->setType(value.toUtf8().constData());
		updateActiveListItem();
	}
}

/* 
 * public slot
 */
void FieldContainerViewCtl::partCardinalityChanged(int index)
{
	if (_activePart) {
		_activePart->setCardinality(index);
		updateActiveListItem();
	}	
}

/* 
 * public slot
 */
void FieldContainerViewCtl::saveSlot()
{
	sync();

	if (_fieldContainer.name())
		_fieldContainer.writeDesc();
	else
		QMessageBox::information ( this, "No Name",
															 "The current FieldContainer doesn't have a name\n"
															 "Can't write the fcd file"
															 );
}

/* 
 * public slot
 */
void FieldContainerViewCtl::partVisibilityChanged(int index)
{
	if (_activePart) {
		_activePart->setVisibility(index);
		updateActiveListItem();
	}
}

/* 
 * public slot
 */
void FieldContainerViewCtl::writeFieldContainerBaseSlot()
{
	char file[1024];

	sync();

	if (_fieldContainer.name()) {
		QString s( QFileDialog::getExistingDirectory( this, QString("Choose directory to save base files to"), QDir::current().path() ) );

		if ( s.isEmpty() )
			return;

		sprintf( file,"%s/%s%sFields.%s", s.toUtf8().constData(), 
                    _fieldContainer.filePrefix(), 
						        _fieldContainer.name(), 
                    _fieldContainer.decFileSuffix());
                    
		if (!_fieldContainer.writeCodeFields(file))
    {
            QMessageBox::warning ( this, "Write error",
														 "Couldn't write the fields file ");
            return;
    }

		sprintf( file,"%s/%s%sBase.%s", s.toUtf8().constData(),
                _fieldContainer.filePrefix(),
						    _fieldContainer.name(), 
                _fieldContainer.decFileSuffix());
                    
		if (!_fieldContainer.writeBaseCodeDec(file))
    {
            QMessageBox::warning ( this, "Write error",
														 "Couldn't write the dec file ");
            return;
    }

		sprintf( file,"%s/%s%sBase.%s", s.toUtf8().constData(),
                _fieldContainer.filePrefix(),
						    _fieldContainer.name(), 
                _fieldContainer.inlFileSuffix());
                    
		if (!_fieldContainer.writeBaseCodeInl(file))
    {
            QMessageBox::warning ( this, "Write error",
														 "Couldn't write the inl file ");
            return;
    }

		sprintf( file,"%s/%s%sBase.%s", s.toUtf8().constData(),
                _fieldContainer.filePrefix(),
						    _fieldContainer.name(), 
                _fieldContainer.impFileSuffix());
                    
		if (!_fieldContainer.writeBaseCodeImp(file))
    {
            QMessageBox::warning ( this, "Write error",
														 "Couldn't write the imp file ");
            return;
    }

	  char bdec[1024],binl[1024],bimp[1024],fields[1024];

		sprintf( fields,"%s/%s%sDecoratorFields.%s", s.toUtf8().constData(),
                    _fieldContainer.filePrefix(), 
						        _fieldContainer.name(), 
                    _fieldContainer.decFileSuffix());
		sprintf( bdec,"%s/%s%sDecoratorBase.%s", s.toUtf8().constData(),
                _fieldContainer.filePrefix(),
						    _fieldContainer.name(), 
                _fieldContainer.decFileSuffix());
		sprintf( binl,"%s/%s%sDecoratorBase.%s", s.toUtf8().constData(),
                _fieldContainer.filePrefix(),
						    _fieldContainer.name(), 
                _fieldContainer.inlFileSuffix());
		sprintf( bimp,"%s/%s%sDecoratorBase.%s", s.toUtf8().constData(),
                _fieldContainer.filePrefix(),
						    _fieldContainer.name(), 
                _fieldContainer.impFileSuffix());
                    
		if (!_fieldContainer.writeDecoratorBase(bdec,binl,bimp,fields))
    {
            QMessageBox::warning ( this, "Write error",
														 "Couldn't write the decorator files ");
            return;
    }
	}
	else
		QMessageBox::warning ( this, "Write error",
													 "No node name");

						
}

/* 
 * public slot
 */
void FieldContainerViewCtl::writeFieldContainerSlot()
{
	char file[6][1024];

	sync();

	if (_fieldContainer.name()) {
		QString s( QFileDialog::getExistingDirectory( this, QString("Choose directory to save files to"), QDir::current().path() ) );

		if ( s.isEmpty() )
			return;

		sprintf( file[0],"%s/%s%s.%s", s.toUtf8().constData(),
                _fieldContainer.filePrefix(),
						    _fieldContainer.name(), 
                _fieldContainer.decFileSuffix());

		sprintf( file[1],"%s/%s%s.%s", s.toUtf8().constData(),
                _fieldContainer.filePrefix(),
						    _fieldContainer.name(), 
                _fieldContainer.inlFileSuffix());

		sprintf( file[2],"%s/%s%s.%s", s.toUtf8().constData(),
                _fieldContainer.filePrefix(),
						    _fieldContainer.name(), 
                _fieldContainer.impFileSuffix());

		sprintf( file[3],"%s/%s%sDecorator.%s", s.toUtf8().constData(),
                _fieldContainer.filePrefix(),
						    _fieldContainer.name(), 
                _fieldContainer.decFileSuffix());
		sprintf( file[4],"%s/%s%sDecorator.%s", s.toUtf8().constData(),
                _fieldContainer.filePrefix(),
						    _fieldContainer.name(), 
                _fieldContainer.inlFileSuffix());
		sprintf( file[5],"%s/%s%sDecorator.%s", s.toUtf8().constData(),
                _fieldContainer.filePrefix(),
						    _fieldContainer.name(), 
                _fieldContainer.impFileSuffix());

				
		// Check if the files already exist and don't overwrite them
		QString ex;		 
    for(int i = 0; i < 6; i++)
    {
		    QFile f( file[i] );
		    if ( f.exists() )
		    {
			    ex += file[i];
			    ex += "\n";
		    }
    } 

		if ( ex.length() )
		{
			QString mes("Some files already exist:\n");
			mes += ex;
			mes += "Are you sure you want to overwrite them?";
			
			if ( QMessageBox::warning ( this, "Overwrite Warning", mes, 
					"Yes", "No", NULL, 0, 1) )
				return;
		}
		
                     
		if (!_fieldContainer.writeCodeDec(file[0]))
    {
            QMessageBox::warning ( this, "Write error",
														 "Couldn't write the dec file ");
            return;
    }
                    
		if (!_fieldContainer.writeCodeInl(file[1]))
    {
            QMessageBox::warning ( this, "Write error",
														 "Couldn't write the inl file ");
            return;
    }
                    
		if (!_fieldContainer.writeCodeImp(file[2]))
    {
            QMessageBox::warning ( this, "Write error",
														 "Couldn't write the imp file ");
            return;
    }
                   
		if (!_fieldContainer.writeDecoratorCode(file[3],file[4],file[5]))
    {
            QMessageBox::warning ( this, "Write error",
														 "Couldn't write the decorator files ");
            return;
    }
	}
	else
		QMessageBox::warning ( this, "Write error",
													 "No node name");

						
}

/* 
 * public slot
 */
void FieldContainerViewCtl::aboutSlot()
{
	QMessageBox::about ( this, "About",
											 "OpenSG FieldContainer Description File Editor\n\n"
											 "http://www.opensg.org"
											 );
}

/* 
 * public slot
 */
void FieldContainerViewCtl::saveAsSlot()
{
	QString fN (QFileDialog::getSaveFileName ( this, "FieldContainer Description File (*.fcd)" ) );
	if (!fN.isEmpty())
    {
        setWindowTitle(tr("OpenSG Field Container Editor : %1 ").arg(fN));
		_fieldContainer.writeDesc(fN.toUtf8().constData());
    }
}

/* 
 * public slot
 */
void FieldContainerViewCtl::loadParentSlot()
{
	char fileName[256];

	if (_fieldContainer.parentFieldContainer()) {
		sprintf(fileName,"%s.fcd", _fieldContainer.parentFieldContainer());
		loadFieldContainer(fileName);
	}
	else
		QMessageBox::information ( this, "No Parent",
													 "The current FieldContainer doesn't have a parent node"
													 );
													 
}

void FieldContainerViewCtl::loadFieldTypesSlot(void)
{
	int i;
	QString fN (QFileDialog::getOpenFileName ( this, "Type list file (*.ftd)" ) );

	if (!fN.isEmpty()) {
		newSlot();
		Field::loadFieldTypeList(fN.toUtf8().constData());
		partTypeCombo->clear();
		for (i = 0; true; i++) 
			if (Field::typeStr(i))
				partTypeCombo->addItem(QString(Field::typeStr(i)));
			else
				break;
	}
}

void FieldContainerViewCtl::resetFieldTypesSlot(void)
{
	int i;

	Field::loadDefaultFieldTypeList();
	partTypeCombo->clear();
	for (i = 0; true; i++) 
		if (Field::typeStr(i))
			partTypeCombo->addItem(QString(Field::typeStr(i)));
		else
			break;
}

void FieldContainerViewCtl::newSlot(void)
{
    QString emptyStr("");
    _fieldContainer.clear();
    
    setWindowTitle( tr( "OpenSG Field Container Editor " ) );
    
    nodeNameInput->blockSignals(true);
    nodeNameInput->setText(emptyStr);
    nodeNameInput->blockSignals(false);
    
    nodeParentInput->blockSignals(true);
    nodeParentInput->setText(emptyStr);
    nodeParentInput->blockSignals(false);
    
    nodeLibraryInput->blockSignals(true);
    nodeLibraryInput->setText(emptyStr);
    nodeLibraryInput->blockSignals(false);
    
    pointerFieldTypesCombo->blockSignals(true);
    pointerFieldTypesCombo->setCurrentIndex(0);
    pointerFieldTypesCombo->blockSignals(false);
    
    nodeDescriptionInput->blockSignals(true);
    nodeDescriptionInput->setText(emptyStr);
    nodeDescriptionInput->blockSignals(false);
    
    partFrame->setEnabled(false);
    _activePartIndex = -1;
    _activePart = 0;
    updateActivePartView();			
    updateList();
}

void FieldContainerViewCtl::pointerFieldTypesChanged(int index)
{
	_fieldContainer.setPointerFieldTypes(index);
}

void FieldContainerViewCtl::structureChanged(int index)
{
	std::cerr << "set abstract: " << index << std::endl;

	_fieldContainer.setAbstract(index);
}

void FieldContainerViewCtl::sysCompChanged(int index)
{
	_fieldContainer.setSystemComponent(index);
	
	if ( index )
	{
		parentSysComp->setChecked( true );
		parentSysComp->setDisabled( true );
	}
	else
	{
		parentSysComp->setDisabled( false );
	}
}

void FieldContainerViewCtl::parentSysCompChanged(int index)
{
	_fieldContainer.setParentSystemComponent(index);
}

void FieldContainerViewCtl::decoratableSwitch_stateChanged(int index)
{
	_fieldContainer.setDecoratable(index);
	
    if (index && _fieldContainer.pointerFieldTypes() == 0)
    {
        _fieldContainer.setPointerFieldTypes(1);
        pointerFieldTypesCombo->setCurrentIndex(1);
    }
}

void FieldContainerViewCtl::localIncludesChanged(int index)
{
	_fieldContainer.setUseLocalIncludes(index);
}

void FieldContainerViewCtl::partAccessChanged(int index)
{
	if (_activePart) {
		_activePart->setAccess(index);
		updateActiveListItem();
	}
}

void FieldContainerViewCtl::partIncludeChanged(const QString &str)
{
	if (_activePart) {
		_activePart->setHeader(str.toUtf8().constData());
		updateActiveListItem();
	}
}

void FieldContainerViewCtl::partDefaultHeaderChanged(const QString &str)
{
	if (_activePart) {
		_activePart->setDefaultHeader(str.toUtf8().constData());
		updateActiveListItem();
	}
}



