/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000,2001 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <boost/python/errors.hpp> 
#include <Python.h>
#include <Nodes.h>
#include <OSGSimpleGeometry.h>
#include <OSGSceneFileHandler.h>

// Basic Node

NodeBase::NodeBase()
{
    _node = osg::Node::create();
}

NodeBase::NodeBase(const NodeBase &copy) : _node(copy._node)
{
}

NodeBase::NodeBase(osg::NodePtr node) : _node(node)
{
}

NodeBase::NodeBase(osg::NodeCorePtr core)
{
    _node = osg::Node::create();
    
    beginEditCP(_node, osg::FieldBits::AllFields);
    _node->setCore(core);
    endEditCP(_node, osg::FieldBits::AllFields);    
}

NodeBase::~NodeBase()
{    
}

const char *NodeBase::getType(void)
{
    if(_node->getCore() == osg::NullFC)
        return "<uncored>";
    
    return _node->getCore()->getType().getCName();
}

void NodeBase::dump(void)
{
    _node->dump();
}

NodeBase NodeBase::clone(void)
{
    return NodeBase(osg::deepCloneTree(_node));
}


NodeIterator NodeBase::iter(const char * name)
{
    return NodeIterator(*this, name);
}
    
TypedNodeIterator<Geometry> NodeBase::geometries(void)
{
    return TypedNodeIterator<Geometry>(*this);
}
    
TypedNodeIterator<Transform> NodeBase::transforms(void)
{
    return TypedNodeIterator<Transform>(*this);
}

// Group Node

Group::Group() : NodeBase()
{
    _group = osg::Group::create();
    
    osg::beginEditCP(_node, osg::FieldBits::AllFields);
    _node->setCore(_group);
    osg::endEditCP(_node, osg::FieldBits::AllFields);
}

Group::Group(const Group &group) : NodeBase(group)
{
    _group = group._group;
}

Group::Group(osg::NodePtr node) : NodeBase(node)
{
    _group = osg::GroupPtr::dcast(node->getCore());
   
    osg::beginEditCP(_node, osg::FieldBits::AllFields);
    _node->setCore(_group);
    osg::endEditCP(_node, osg::FieldBits::AllFields);
}

Group::Group(osg::GroupPtr group) : NodeBase()
{
    osg::beginEditCP(_node, osg::FieldBits::AllFields);
    _node->setCore(_group);
    osg::endEditCP(_node, osg::FieldBits::AllFields);
}

Group::~Group()
{    
}


void Group::addChild(NodeBase &child) 
{
    osg::beginEditCP(_node, osg::FieldBits::AllFields);
    _node->addChild(child.getNode());
    osg::endEditCP(_node, osg::FieldBits::AllFields);
}


void Group::subChild(NodeBase &child) 
{
    osg::beginEditCP(_node, osg::FieldBits::AllFields);
    _node->subChild(child.getNode());
    osg::endEditCP(_node, osg::FieldBits::AllFields);
}

// Transform Node

Transform::Transform() 
{
    _transform = osg::Transform::create();
    
    osg::beginEditCP(_node, osg::FieldBits::AllFields);
    _node->setCore(_transform);
    osg::endEditCP(_node, osg::FieldBits::AllFields);
}

Transform::Transform(const Transform &trans) : Group(trans)
{
    _transform = trans._transform;
}

Transform::Transform(osg::NodePtr node) : Group(node)
{
    _transform = osg::TransformPtr::dcast(node->getCore());
   
    osg::beginEditCP(_node, osg::FieldBits::AllFields);
    _node->setCore(_transform);
    osg::endEditCP(_node, osg::FieldBits::AllFields);
}

Transform::Transform(osg::TransformPtr transform) : Group()
{
    osg::beginEditCP(_node, osg::FieldBits::AllFields);
    _node->setCore(_transform);
    osg::endEditCP(_node, osg::FieldBits::AllFields);
}

Transform::~Transform()
{    
}


void Transform::setTranslation(osg::Real32 x, osg::Real32 y, osg::Real32 z) 
{
    osg::Matrix m;
    
    m.setTransform(osg::Vec3f(x,y,z));
    
    osg::beginEditCP(_transform, osg::FieldBits::AllFields);
    _transform->setMatrix(m);
    osg::endEditCP(_transform, osg::FieldBits::AllFields);
}


// Geometry Node

Geometry::Geometry() 
{
    _geometry = osg::Geometry::create();
    
    osg::beginEditCP(_node, osg::FieldBits::AllFields);
    _node->setCore(_geometry);
    osg::endEditCP(_node, osg::FieldBits::AllFields);
}

Geometry::Geometry(const Geometry &geo) : NodeBase(geo)
{
    _geometry = geo._geometry;
}

Geometry::Geometry(osg::NodePtr node) : NodeBase(node)
{
    _geometry = osg::GeometryPtr::dcast(node->getCore());
   
    osg::beginEditCP(_node, osg::FieldBits::AllFields);
    _node->setCore(_geometry);
    osg::endEditCP(_node, osg::FieldBits::AllFields);
}

Geometry::Geometry(osg::GeometryPtr geometry) : NodeBase()
{
    osg::beginEditCP(_node, osg::FieldBits::AllFields);
    _node->setCore(_geometry);
    osg::endEditCP(_node, osg::FieldBits::AllFields);
}

Geometry::~Geometry()
{    
}


void Geometry::translate(osg::Real32 x, osg::Real32 y, osg::Real32 z) 
{
    osg::Vec3f d(x,y,z);
    osg::Pnt3f p;
    osg::GeoPositionsPtr pos = _geometry->getPositions();
    
    osg::beginEditCP(_geometry, osg::Geometry::PositionsFieldMask);
    
    for(osg::UInt32 i = 0; i < pos->size(); ++i)
    {
        pos->getValue(p, i);
        p += d;
        pos->setValue(p,i);
    }

    osg::endEditCP(_geometry, osg::Geometry::PositionsFieldMask);
}

void Geometry::scale(osg::Real32 x, osg::Real32 y, osg::Real32 z) 
{
    osg::Pnt3f p;
    osg::GeoPositionsPtr pos = _geometry->getPositions();
    
    osg::beginEditCP(_geometry, osg::Geometry::PositionsFieldMask);
    
    for(osg::UInt32 i = 0; i < pos->size(); ++i)
    {
        pos->getValue(p, i);
        p[0] *= x;
        p[1] *= y;
        p[2] *= z;
        pos->setValue(p,i);
    }

    osg::endEditCP(_geometry, osg::Geometry::PositionsFieldMask);
}


void Geometry::merge(Geometry &geo) 
{
    if(_geometry->isMergeable(geo._geometry))
    {
        _geometry->merge(geo._geometry);
    }
    else
    {
        FWARNING(("Can't merge geometries!\n"));
    }
}


void Geometry::setDlistCache(bool cache) 
{

}


// Node Iterator

NodeIterator::NodeIterator(void)
{
}

NodeIterator::NodeIterator(NodeBase &start, const char *name)
{
    _stack.push_back(start.getNode());
    
    _type = osg::FieldContainerFactory::the()->findType(name);
    
}

NodeIterator::NodeIterator(const NodeIterator &copy) : 
    _type(copy._type), _stack(copy._stack)
{
}

NodeIterator::~NodeIterator()
{
}

NodeIterator NodeIterator::__iter__(void)
{
    return *this;
}

NodeBase NodeIterator::next(void)
{
    while(!_stack.empty())
    {
        osg::NodePtr act = _stack.back();

        _stack.pop_back();

        for(osg::UInt32 i = 0; i < act->getNChildren(); ++i)
            _stack.push_back(act->getChild(i));

        if(act->getCore()->getType().isDerivedFrom(*_type))
        {
            return NodeBase(act);
        }    
    }
   
    PyErr_SetString(PyExc_StopIteration, "Out of Nodes");   
    boost::python::throw_error_already_set();
}


template <class type>    
TypedNodeIterator<type>::TypedNodeIterator(void) : NodeIterator()
{
    _type = osg::FieldContainerFactory::the()->findType(type::getCoreName());
}

template <class type>    
TypedNodeIterator<type>::TypedNodeIterator(NodeBase &start) : 
    NodeIterator(start, type::getCoreName())
{
}

template <class type>    
TypedNodeIterator<type>::TypedNodeIterator(const TypedNodeIterator<type> &copy)
: NodeIterator(copy)
{
}
    
template <class type>    
TypedNodeIterator<type>::~TypedNodeIterator()
{
}
    
template <class type>       
TypedNodeIterator<type> TypedNodeIterator<type>::__iter__(void)
{
    return *this;
}
    
template <class type>    
type TypedNodeIterator<type>::next(void)
{
    return type(NodeIterator::next().getNode());
}


template class TypedNodeIterator<Geometry>;
template class TypedNodeIterator<Transform>;

// Functions


Geometry makeBox(osg::Real32 x, osg::Real32 y, osg::Real32 z,
                 osg::UInt16 hor, osg::UInt16 vert, osg::UInt16 depth)
{
    Geometry g(osg::makeBox(x,y,z,hor,vert,depth));
    
    return g;
}

Geometry makeTorus(osg::Real32 inner, osg::Real32 outer,
                   osg::UInt16 sides, osg::UInt16 rings)
{
    Geometry g(osg::makeTorus(inner,outer,sides,rings));
    
    return g;
}

Geometry makeSphere(osg::UInt16 latres, osg::UInt16 longres, 
                    osg::Real32 radius)
{
    Geometry g(osg::makeLatLongSphere(latres,longres,radius));
    
    return g;
}


Geometry makePlane(osg::Real32 x, osg::Real32 y,
                   osg::UInt16 hor, osg::UInt16 vert)
{
    Geometry g(osg::makePlane(x, y, hor, vert));
    
    return g;
}

NodeBase makeShared(NodeBase &node)
{
    return NodeBase(node.getNode()->getCore());
}

void addRef(NodeBase &node)
{
    addRefCP(node.getNode());
}

void subRef(NodeBase &node)
{
    subRefCP(node.getNode());
}

NodeBase loadScene(char *filename)
{
   osg:: NodePtr node = osg::SceneFileHandler::the().read(filename);
    
    return NodeBase(node);
}
