/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000,2001 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <OSGGLUT.h>
#include <OSGMatrixUtility.h>

#include "Test.h"

Test::Test(void) :
    _win(NULL), _scene(osg::NullFC), _near(0), _far(0),
    _froms(), _oris(), _fovs(), 
    _minTime(-1), _nFrames(0), _headlight(true),
    _statsLevel(0), _time(0), _nRenderedFrames(0), _stats(),
    _verbose(false)
{
}

Test::~Test()
{
}

// Set up scene characteristics

void Test::setScene(NodeBase &scene)
{
    _scene = scene.getNode();
}

void Test::setScene(osg::NodePtr scene)
{
    _scene = scene;
}

void Test::setWindow(TestWindow &win)
{
    _win = &win;
}

void Test::setHeadlight(bool on)
{
    _headlight = on;
}
     
void Test::setNearFar(osg::Real32 n, osg::Real32 f)
{
    _near = n;
    _far = f;
}

// Setup Test Frames

// clear all the data
void Test::clear(void)
{
    _froms.clear();
    _oris.clear();
    _fovs.clear();
}

// MinTime takes precedence, the path is repeated fully until the time is
// reached

void Test::setNFrames(osg::UInt32 nframes)
{
    _nFrames = nframes;
}
   
void Test::setMinTime(osg::Real32 minTime)
{
    _minTime = minTime;
}

void Test::addFromAtUp(osg::Pnt3f from, osg::Pnt3f at, osg::Vec3f up)
{
    osg::Matrix m;
    
    osg::MatrixLookAt(m, from, at, up);
                           
    _froms.push_back(from);
    _oris.push_back(osg::Quaternion(m));
}

void Test::addFromAtUp(osg::Real32 fromx, osg::Real32 fromy, osg::Real32 fromz, 
                       osg::Real32 atx,   osg::Real32 aty,   osg::Real32 atz, 
                       osg::Real32 upx,   osg::Real32 upy,   osg::Real32 upz)
{
    addFromAtUp(osg::Pnt3f(fromx, fromy, fromz), 
                osg::Pnt3f(atx, aty, atz),
                osg::Vec3f(upx, upy, upz));
}

// Define Path from VRML-Style Positon/Quaternion Strings
void Test::addFromOri(osg::Char8 *from, osg::Char8 *ori)
{
    osg::Char8 *f = from, *o = ori;
    osg::UInt32 frames = 0;
    
    while(f && *f && o && *o)
    {
        osg::Pnt3f v;
        osg::Quaternion q;
        
        const osg::Char8 *grmbl=f;
#if defined(OSG_VERSION) && OSG_VERSION >= 010300
        if (!osg::FieldDataTraits<osg::Pnt3f>::getFromString(v, grmbl))
#else
        if (osg::FieldDataTraits<osg::Pnt3f>::getFromString(v, grmbl))
#endif
        {
            FWARNING(("Test::addFromOri: error reading from: '%s'!\n",
                        f));
            return;
        }
        
        grmbl=o;
        if (!osg::FieldDataTraits<osg::Quaternion>::getFromString(q, grmbl))
        {
            FWARNING(("Test::addFromOri: error reading ori: '%s'!\n",
                        o));
            return;
        }
        
        _froms.push_back(v);
        _oris.push_back(q);             
        
        f = strchr(f, ',');
        if(f)
            ++f;
        
        o = strchr(o, ',');
        if(o)
            ++o;
    }
}

// Make a rotational path around the whole model
void Test::makeOrbit(osg::Real32 upx, osg::Real32 upy, osg::Real32 upz)
{
    if(_scene == osg::NullFC)
    {
        FWARNING(("Test::makeOrbit: need scene!\n"));
        return;
    }
    
    _scene->updateVolume();
    
    osg::DynamicVolume volume;
    _scene->getWorldVolume(volume);
    
    osg::Pnt3f center;
    volume.getCenter(center);
    
    osg::Real32 dia = (volume.getMax() - volume.getMin()).length();
    
    osg::Vec3f up(upx,upy,upz);
    osg::Vec3f dir, right;
    
    up.normalize();
    
    dir = up.cross(osg::Vec3f(1,0,0));
    if(dir.squareLength() < osg::Eps)
    {
        dir = up.cross(osg::Vec3f(0,1,0));
        
        if(dir.squareLength() < osg::Eps)
        {
            dir = up.cross(osg::Vec3f(0,0,1));
            if(dir.squareLength() < osg::Eps)
            {
                up.setValues(0,1,0);
                dir.setValues(0,0,1);
            }
        }
    }
    
    dir.normalize();
    right = dir.cross(up);
    up = right.cross(dir);
    
    dir *= dia;
    right *= dia;
    
    _froms.clear();
    _oris.clear();
    
    addFromAtUp(center + dir * osg::Sqrt2                     , center, up);
    addFromAtUp(center + dir              + right             , center, up);
    
    addFromAtUp(center                    + right * osg::Sqrt2, center, up);
    addFromAtUp(center - dir              + right             , center, up);
    
    addFromAtUp(center - dir * osg::Sqrt2                     , center, up);
    addFromAtUp(center - dir              - right             , center, up);
    
    addFromAtUp(center                    - right * osg::Sqrt2, center, up);
    addFromAtUp(center + dir              - right             , center, up);
    
    addFromAtUp(center + dir * osg::Sqrt2                     , center, up);
}

// Make a pirouette inside the model
void Test::makePirouette(osg::Real32 upx,   osg::Real32 upy,   osg::Real32 upz)
{
    if(_scene == osg::NullFC)
    {
        FWARNING(("Test::makePirouette: need scene!\n"));
        return;
    }
    
    _scene->updateVolume();
    
    osg::DynamicVolume volume;
    _scene->getWorldVolume(volume);
    
    osg::Pnt3f center;
    volume.getCenter(center);
    
    osg::Real32 dia = (volume.getMax() - volume.getMin()).length() * .5;
    
    osg::Vec3f up(upx,upy,upz);
    osg::Vec3f dir, right;
    
    up.normalize();
    
    dir = up.cross(osg::Vec3f(1,0,0));
    if(dir.squareLength() < osg::Eps)
    {
        dir = up.cross(osg::Vec3f(0,1,0));
        
        if(dir.squareLength() < osg::Eps)
        {
            dir = up.cross(osg::Vec3f(0,0,1));
            if(dir.squareLength() < osg::Eps)
            {
                up.setValues(0,1,0);
                dir.setValues(0,0,1);
            }
        }
    }
    
    dir.normalize();
    right = dir.cross(up);
    up = right.cross(dir);
    
    dir *= dia;
    right *= dia;
    
    _froms.clear();
    _oris.clear();
     
    addFromAtUp(center, center + dir * osg::Sqrt2                     , up);
    addFromAtUp(center, center + dir              + right             , up);
    
    addFromAtUp(center, center                    + right * osg::Sqrt2, up);
    addFromAtUp(center, center - dir              + right             , up);
    
    addFromAtUp(center, center - dir * osg::Sqrt2                     , up);
    addFromAtUp(center, center - dir              - right             , up);
    
    addFromAtUp(center, center                    - right * osg::Sqrt2, up);
    addFromAtUp(center, center + dir              - right             , up);
    
    addFromAtUp(center, center + dir * osg::Sqrt2                     , up);
   
}

// add a FOV to animate
void Test::addFov(osg::Real32 fov)
{
    _fovs.push_back(fov);
}

// Run Test
void Test::setStatistics(osg::UInt16 level)
{
    _statsLevel = level;
}

osg::UInt16 Test::getStatistics(void)
{
    return _statsLevel;
}

void Test::setVerbose(bool verbose)
{
    _verbose = verbose;
}

// little helper function that runs the test loop. The main reason for this
// function is being able to profile it, without the initialisation.

void Test::runLoop( std::vector<osg::Matrix> &views,
                    std::vector<osg::Real32> &fovs)
{
    osg::Time start, stop;
    osg::SimpleSceneManager *ssm = _win->getSSM();

    start = osg::getSystemTime();
    
    do
    {        
        if(_statsLevel)
        {
            osg::UInt32 ss = _stats.size();

            _stats.resize(_stats.size() + views.size());

            if(_statsLevel > 1)
            {
                for(osg::UInt32 i = ss; i < _stats.size(); ++i)
                {
                    _stats[i].getElem(osg::Geometry::statNTriangles);
                }
            }
        }
        
        for(osg::UInt32 i = 0; i < views.size(); ++i, ++_nRenderedFrames)
        {
            if(_verbose)
                SWARNING << "Test::run: Frame " << i << " ("
                         << _nRenderedFrames << ") fov " 
                         << fovs[i] << ", view" << std::endl << views[i] 
                         << std::endl;
            _win->setCamera(views[i]);
            _win->setFov(fovs[i]);
            
            if(_statsLevel)
            {
                ssm->getAction()->setStatistics(&_stats[_nRenderedFrames]);
            }
            
            _win->redraw();
        }
    }
    while(_minTime > 0 && osg::getSystemTime() - start < _minTime);
   
    _win->finish();
    
    stop = osg::getSystemTime();

    _time = stop - start;  

    if(_statsLevel)
    {
        ssm->getAction()->setStatistics(NULL);
    }
}

void Test::run(void)
{
    if(!_win || !_win->isOpen())
    {
        FWARNING(("Test::run: window not ready!\n"));
        return;
    }
    if(!_scene)
    {
        FWARNING(("Test::run: no scene!\n"));
        return;
    }
    if(_froms.empty())
    {
        FWARNING(("Test::run: no views!\n"));
        return;
    }
    if(_oris.empty())
    {
        FWARNING(("Test::run: no views!\n"));
        return;
    }
    if(_froms.size() != _oris.size())
    {
        FWARNING(("Test::run: _froms.size() != _oris.size()!\n"));
        return;        
    }
    
    if(_fovs.empty())
    {
        FWARNING(("Test::run: no fovs!\n"));
        return;
    }
    
    std::vector<osg::Matrix> views;
    std::vector<osg::Real32> fovs;
   
    expandData(views, fovs);
    
    osg::Time start, stop;
    osg::SimpleSceneManager *ssm = _win->getSSM();
    
    _win->setScene(_scene);
    ssm->setHeadlight(_headlight);
   
    _stats.clear();
 
    // render two start frames to prime caches
    
    _win->showAll();
    _win->redraw();
    
    if(_near > 0 && _far > 0)
    {
        _win->setNearFar(_near, _far);
        if(_verbose)
            FWARNING(("Test::run: near=%f, far=%f\n", _near, _far));
    }
      
    _nRenderedFrames = 0;
    
    _win->finish();

    runLoop(views, fovs);
}


Image Test::snapshot(osg::UInt32 frame)
{
    if(!_win || !_win->isOpen())
    {
        FWARNING(("Test::snapshot: window not ready!\n"));
        return Image();
    }
    if(!_scene)
    {
        FWARNING(("Test::snapshot: no scene!\n"));
        return Image();
    }
    if(_froms.empty())
    {
        FWARNING(("Test::snapshot: no views!\n"));
        return Image();
    }
    if(_oris.empty())
    {
        FWARNING(("Test::snapshot: no views!\n"));
        return Image();
    }
    if(_froms.size() != _oris.size())
    {
        FWARNING(("Test::snapshot: _froms.size() != _oris.size()!\n"));
        return Image();        
    }
    
    if(_fovs.empty())
    {
        FWARNING(("Test::snapshot: no fovs!\n"));
        return Image();
    }
    
    std::vector<osg::Matrix> views;
    std::vector<osg::Real32> fovs;
   
    expandData(views, fovs);
    
    if(frame >= views.size())
    {
        FWARNING(("Test::snapshot: frame >= views.size()!\n"));
        return Image();            
    }
    
    osg::SimpleSceneManager *ssm = _win->getSSM();
    
    _win->setScene(_scene);
    ssm->setHeadlight(_headlight);

    _win->setCamera(views[frame]);
    _win->setFov(fovs[frame]);
    
    ssm->getAction()->setStatistics(NULL); // Don't want this to count
    
    osg::ImagePtr img;
    
    img = _win->snapshot();
    
    return Image(img);
}

// Get Results

osg::Real32 Test::getFPS(void)
{
    return _nRenderedFrames / _time;
}

osg::Real32 Test::getTime(void)
{
    return _time;
}

osg::UInt32 Test::getNRenderedFrames(void)
{
    return _nRenderedFrames;
}

osg::Real64 Test::getStatValue(osg::Char8 *name, osg::UInt32 frame)
{
    if(_stats.size() <= frame)
    {
        FWARNING(("Test::getStatValue: no statistics for frame %d!\n", frame));
        return -1;
    }
    
    osg::Real64 val;
    std::string n(name);
    
#if defined(OSG_VERSION) && OSG_VERSION >= 13
    if(!_stats[frame].getValue(n, val))
    {
        FWARNING(("Test::getStatValue: no statistics value '%s'!\n", name));
        return -1;
    }
#else
    osg::StatElemDescBase *desc = osg::StatElemDescBase::findDescByName(name);
    if(!desc)
    {
        FWARNING(("Test::getStatValue: name not known '%s'!\n", name));
        return -1;
    }
    osg::StatElem *el = _stats[frame].getElem(*desc, false);
    if(!el)
    {
        FWARNING(("Test::getStatValue: no statistics value '%s'!\n", name));
        return -1;
    }
    val = el->getValue();
#endif
    
    return val;   
}


// expand the animation data to have nFrames elements

void Test::expandData(std::vector<osg::Matrix>& views, std::vector<osg::Real32> &fovs)
{
    views.resize(_nFrames);
    fovs.resize(_nFrames);
    
    // interpolate the values to get _nFrames values
    
    osg::Real32 vscale = (_froms.size() - 1) / (_nFrames - 1.f);
    osg::Real32 fscale = (_fovs.size() - 1) / (_nFrames - 1.f);
    
    for(osg::UInt32 i = 0; i < _nFrames; ++i)
    {
        osg::UInt32 vi = static_cast<osg::UInt32>(osg::osgfloor(i * vscale));
        osg::Real32 v = i * vscale - vi;
        
        if(v < osg::Eps)
        {
            views[i].setTransform(_froms[vi].subZero(), _oris[vi]);
        }
        else
        {
            osg::Quaternion q;           
            q.slerpThis(_oris[vi], _oris[vi+1], v);

            osg::Pnt3f p = _froms[vi] + v * (_froms[vi+1] - _froms[vi]);

            views[i].setTransform(p.subZero(), q);
        }
 
        osg::UInt32 fi = static_cast<osg::UInt32>(osg::osgfloor(i * fscale));
        osg::Real32 f = i * fscale - fi;
        
        if(f < osg::Eps)
        {
            fovs[i] = _fovs[fi];
        }
        else
        {
            fovs[i] = _fovs[fi] * (1.f  - f) + _fovs[fi + 1] * f;;
        }
    }   
}
