/*---------------------------------------------------------------------------*\
 *                                OpenSG                                     *
 *                                                                           *
 *                                                                           *
 *             Copyright (C) 2000,2001 by the OpenSG Forum                   *
 *                                                                           *
 *                            www.opensg.org                                 *
 *                                                                           *
 *   contact: dirk@opensg.org, gerrit.voss@vossg.org, jbehr@zgdv.de          *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                License                                    *
 *                                                                           *
 * This library is free software; you can redistribute it and/or modify it   *
 * under the terms of the GNU Library General Public License as published    *
 * by the Free Software Foundation, version 2.                               *
 *                                                                           *
 * This library is distributed in the hope that it will be useful, but       *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU         *
 * Library General Public License for more details.                          *
 *                                                                           *
 * You should have received a copy of the GNU Library General Public         *
 * License along with this library; if not, write to the Free Software       *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
 *                                                                           *
\*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*\
 *                                Changes                                    *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
 *                                                                           *
\*---------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Includes
//---------------------------------------------------------------------------

#include <OSGGLUT.h>
#include <OSGMatrixUtility.h>
#include <setjmp.h>

#include "TestWindow.h"

// This is ultra-ugly, but GLUT gives you no alternative
static jmp_buf jump;


TestWindow::TestWindow(void) :
    _width(-1), _height(-1), _scene(osg::NullFC), _window(osg::NullFC), _ssm(NULL),
    _open(false), _left(0), _right(1), _top(1), _bottom(0),
    _near(0.1), _far(100), _fov(1), _beacon(osg::NullFC)
{
}

TestWindow::~TestWindow()
{
    if(_window != osg::NullFC)
        subRefCP(_window);
    if(_ssm != NULL)
        delete _ssm;
}

void TestWindow::setFullscreen(void)
{
    if(!isOpen())
    {
        _width = -1;
        return;
    }
        
    _width = glutGet(GLUT_SCREEN_WIDTH);
    _height = glutGet(GLUT_SCREEN_HEIGHT);
    
    glutFullScreen();

    if(!setjmp(jump))
        glutMainLoop();
    
    update();
}

void TestWindow::setSize(osg::UInt16 width, osg::UInt16 height)
{
    _width = width;
    _height = height;
    
    if(isOpen())
    {
        glutReshapeWindow(width, height);
    
        if(!setjmp(jump))
            glutMainLoop();
    }
    
    update();
}

void TestWindow::setViewport(osg::Real32 left,   osg::Real32 right, 
                             osg::Real32 bottom, osg::Real32 top   )
{
    _left   = left;
    _right  = right;
    _bottom = bottom;
    _top    = top;

    update();
}

void TestWindow::setCamera(osg::Matrix mat)
{
    osg::beginEditCP(_beacon, osg::FieldBits::AllFields);
    _beacon->setMatrix(mat);
    osg::endEditCP(_beacon, osg::FieldBits::AllFields);
}

void TestWindow::showAll(void)
{
    update();
    
    _ssm->showAll();
    _ssm->redraw();
}

void TestWindow::setNearFar(osg::Real32 n, osg::Real32 f)
{
    osg::beginEditCP(getCamera(), osg::FieldBits::AllFields);
    getCamera()->setNear(n);
    getCamera()->setFar(f);
    osg::endEditCP(getCamera(), osg::FieldBits::AllFields);
}

void TestWindow::setFov(osg::Real32 fov)
{
    osg::beginEditCP(getCamera(), osg::FieldBits::AllFields);
    getCamera()->setFov(fov);
    osg::endEditCP(getCamera(), osg::FieldBits::AllFields);  
}

void TestWindow::redraw(void)
{
//    _ssm->redraw();
    _window->render(dynamic_cast<osg::RenderAction*>(_ssm->getAction()));
}

osg::ImagePtr TestWindow::snapshot(void)
{
//    _ssm->redraw();

    osg::ViewportPtr port = _ssm->getWindow()->getPort(0);

    osg::beginEditCP(port, osg::FieldBits::AllFields);
    port->getForegrounds().push_back(_grabber);
    osg::endEditCP(port, osg::FieldBits::AllFields);
    
    osg::ImagePtr img = osg::Image::create();
    osg::beginEditCP(img, osg::FieldBits::AllFields);
    img->setPixelFormat(osg::Image::OSG_RGB_PF);
    osg::endEditCP(img, osg::FieldBits::AllFields);
    
    osg::beginEditCP(_grabber, osg::FieldBits::AllFields);
    _grabber->setImage(img);
    osg::endEditCP(_grabber, osg::FieldBits::AllFields);
    
    _window->render(dynamic_cast<osg::RenderAction*>(_ssm->getAction()));
    
    osg::beginEditCP(port, osg::FieldBits::AllFields);
    port->getForegrounds().erase(port->getForegrounds().end()-1);
    osg::endEditCP(port, osg::FieldBits::AllFields);
    
    return img;
}

void TestWindow::finish(void)
{    
    glFinish();
    
    osg::UInt32 dummy;
    glReadPixels(0, 0, 1, 1, GL_RGB, GL_UNSIGNED_BYTE, &dummy);
}

void TestWindow::setCamera(osg::Real32 fromx, osg::Real32 fromy, osg::Real32 fromz, 
                           osg::Real32 atx,   osg::Real32 aty,   osg::Real32 atz, 
                           osg::Real32 upx,   osg::Real32 upy,   osg::Real32 upz)
{
    osg::Matrix m;
    
    osg::MatrixLookAt(m, fromx, fromy, fromz, 
                           atx,   aty,   atz, 
                           upx,   upy,   upz);
    setCamera(m);
}

static void display(void)
{
    // Make sure the window is really open...
    glClear(GL_COLOR_BUFFER_BIT);

    glFinish();
    
    osg::UInt32 dummy;
    glReadPixels(0, 0, 1, 1, GL_RGB, GL_UNSIGNED_BYTE, &dummy);
}

static void idle(void)
{
    longjmp(jump, 0);
}

void TestWindow::open(void)
{
    static bool inited = false;
    
    if(isOpen())
        return;
        
    if(!inited)
    {
        int argc = 2;
        char *argv[2]={"OpenSG", "Benchmarks"};
        
        glutInit(&argc, argv);
        glutInitDisplayMode(GLUT_RGB | GLUT_DEPTH | GLUT_DOUBLE);
        inited = true;        
    }

    glutInitWindowPosition(0,0);
    if(_width > 0)
        glutInitWindowSize(_width, _height);
   
    int winid = glutCreateWindow("OpenSG Benchmark");

    if(_width < 0)
    {
        glutFullScreen();
        _width = glutGet(GLUT_SCREEN_WIDTH);
        _height = glutGet(GLUT_SCREEN_HEIGHT);       
    }
    
    glutDisplayFunc(display);
    glutIdleFunc(idle);
    
    update();
    
    _window->setId(winid);
    _window->init();
    
    if(!setjmp(jump))
        glutMainLoop();
    
    _open = true;
}

void TestWindow::close(void)
{
    glutDestroyWindow(_window->getId());
    _open = false;
}  

void TestWindow::update(void)
{  
    if(_window == osg::NullFC)
        _window = osg::GLUTWindow::create();
    
    if(_grabber == osg::NullFC)
    {
        _grabber = osg::GrabForeground::create();
        osg::beginEditCP(_grabber, osg::FieldBits::AllFields);
        _grabber->setAutoResize(true);
        _grabber->setActive(true);       
        osg::endEditCP(_grabber, osg::FieldBits::AllFields);
    }
    
    if(_ssm == NULL)
        _ssm = new osg::SimpleSceneManager;
    
    _ssm->setWindow(_window);
  
    _ssm->setRoot(_scene);

    _beacon = osg::TransformPtr::dcast(
                        getCamera()->getBeacon()->getCore());
    
    _ssm->resize(_width, _height);
   
    for(int i = 0; i < _ssm->getWindow()->getPort().getSize(); ++i)
    {
        osg::ViewportPtr port = _ssm->getWindow()->getPort(i);
        
        osg::beginEditCP(port, osg::FieldBits::AllFields);
        port->setLeft(_left);
        port->setRight(_right);
        port->setBottom(_bottom);
        port->setTop(_top);
        osg::endEditCP(port, osg::FieldBits::AllFields);
    }

}
